package loquebot.memory;

import java.util.HashMap;
import java.util.HashSet;
import java.util.ArrayList;
import java.util.Collection;
import java.util.logging.Logger;

import cz.cuni.pogamut.Client.AgentBody;

import cz.cuni.pogamut.MessageObjects.MessageObject;
import cz.cuni.pogamut.MessageObjects.MessageType;
import cz.cuni.pogamut.MessageObjects.ItemType;

import cz.cuni.pogamut.MessageObjects.Spawn;
import cz.cuni.pogamut.MessageObjects.Item;
import cz.cuni.pogamut.MessageObjects.Ammo;
import cz.cuni.pogamut.MessageObjects.Armor;
import cz.cuni.pogamut.MessageObjects.Health;
import cz.cuni.pogamut.MessageObjects.Weapon;
import cz.cuni.pogamut.MessageObjects.Special;
import cz.cuni.pogamut.MessageObjects.NavPoint;
import cz.cuni.pogamut.MessageObjects.AddWeapon;
import cz.cuni.pogamut.MessageObjects.EndMessage;
import cz.cuni.pogamut.MessageObjects.DeleteFromBatch;


import loquebot.Main;
import loquebot.util.LoqueListener;

/**
 * Responsible for listening to the messages and managing info about map items.
 *
 * @author Juraj Simlovic [jsimlo@matfyz.cz]
 * @version Tested on Pogamut 2 platform version 1.0.5.
 */
public class LoqueMapItems
{
    /** All navpoints. */
    private NavPointsInfo navPoints = new NavPointsInfo ();

    /*========================================================================*/

    /** All items. */
    private ItemsInfo items = new ItemsInfo ();
    /** All weapons. */
    private ItemsInfo weapons = new ItemsInfo ();
    /** All ammos. */
    private ItemsInfo ammos = new ItemsInfo ();
    /** All healths. */
    private ItemsInfo healths = new ItemsInfo ();
    /** All armors. */
    private ItemsInfo armors = new ItemsInfo ();
    /** All specials. */
    private ItemsInfo specials = new ItemsInfo ();

    /*========================================================================*/

    private PickupsInfo pickups = new PickupsInfo ();

    /*========================================================================*/

    /**
     * Retreives list of all visible items.
     * @param pickableOnly If true, the list will be filtered by pickability.
     * An item is considered pickable, when the agent can pick it up.
     * @param greedy If false, the list will be filtered by usefullness.
     * An item is considered useful, when it will change agent's status somehow.
     * @return List of all visible items. Note: Spawned items are included only.
     * @note Returns copies of the item infos to prevent outer modifications.
     */
    public ArrayList<Item> getVisibleItems (boolean pickableOnly, boolean greedy)
    {
        return getItems (items.visible, true, pickableOnly, greedy);
    }

    /**
     * Retreives list of all visible weapons.
     * @param pickableOnly If true, the list will be filtered by pickability.
     * An item is considered pickable, when the agent can pick it up.
     * @param greedy If false, the list will be filtered by usefullness.
     * An item is considered useful, when it will change agent's status somehow.
     * @return List of all visible weapons. Note: Spawned items are included only.
     * @note Returns copies of the weapon infos to prevent outer modifications.
     */
    @SuppressWarnings ("unchecked")
    public ArrayList<Weapon> getVisibleWeapons (boolean pickableOnly, boolean greedy)
    {
        return (ArrayList) getItems (weapons.visible, true, pickableOnly, greedy);
    }

    /**
     * Retreives list of all visible ammos.
     * @param pickableOnly If true, the list will be filtered by pickability.
     * An item is considered pickable, when the agent can pick it up.
     * @param greedy If false, the list will be filtered by usefullness.
     * An item is considered useful, when it will change agent's status somehow.
     * @return List of all visible ammos. Note: Spawned items are included only.
     * @note Returns copies of the ammo infos to prevent outer modifications.
     */
    @SuppressWarnings ("unchecked")
    public ArrayList<Ammo> getVisibleAmmos (boolean pickableOnly, boolean greedy)
    {
        return (ArrayList) getItems (ammos.visible, true, pickableOnly, greedy);
    }

    /**
     * Retreives list of all visible healths.
     * @param pickableOnly If true, the list will be filtered by pickability.
     * An item is considered pickable, when the agent can pick it up.
     * @param greedy If false, the list will be filtered by usefullness.
     * An item is considered useful, when it will change agent's status somehow.
     * @return List of all visible healths. Note: Spawned items are included only.
     * @note Returns copies of the health infos to prevent outer modifications.
     */
    @SuppressWarnings ("unchecked")
    public ArrayList<Health> getVisibleHealths (boolean pickableOnly, boolean greedy)
    {
        return (ArrayList) getItems (healths.visible, true, pickableOnly, greedy);
    }

    /**
     * Retreives list of all visible armors.
     * @param pickableOnly If true, the list will be filtered by pickability.
     * An item is considered pickable, when the agent can pick it up.
     * @param greedy If false, the list will be filtered by usefullness.
     * An item is considered useful, when it will change agent's status somehow.
     * @return List of all visible armors. Note: Spawned items are included only.
     * @note Returns copies of the armor infos to prevent outer modifications.
     */
    @SuppressWarnings ("unchecked")
    public ArrayList<Armor> getVisibleArmors (boolean pickableOnly, boolean greedy)
    {
        return (ArrayList) getItems (armors.visible, true, pickableOnly, greedy);
    }

    /**
     * Retreives list of all visible specials.
     * @param pickableOnly If true, the list will be filtered by pickability.
     * An item is considered pickable, when the agent can pick it up.
     * @param greedy If false, the list will be filtered by usefullness.
     * An item is considered useful, when it will change agent's status somehow.
     * @return List of all visible specials. Note: Spawned items are included only.
     * @note Returns copies of the special infos to prevent outer modifications.
     */
    @SuppressWarnings ("unchecked")
    public ArrayList<Special> getVisibleSpecials (boolean pickableOnly, boolean greedy)
    {
        return (ArrayList) getItems (specials.visible, true, pickableOnly, greedy);
    }

    /*========================================================================*/

    /**
     * Retreives list of all items (both known and thrown).
     * @param pickableOnly If true, the list will be filtered by pickability.
     * An item is considered pickable, when the agent can pick it up.
     * @param greedy If false, the list will be filtered by usefullness.
     * An item is considered useful, when it will change agent's status somehow.
     * @return List of all items. Note: Spawned items are included only.
     * @note Returns copies of the item infos to prevent outer modifications.
     */
    public ArrayList<Item> getAllItems (boolean pickableOnly, boolean greedy)
    {
        return getItems (items.all, true, pickableOnly, greedy);
    }

    /**
     * Retreives list of all weapons (both known and thrown).
     * @param pickableOnly If true, the list will be filtered by pickability.
     * An item is considered pickable, when the agent can pick it up.
     * @param greedy If false, the list will be filtered by usefullness.
     * An item is considered useful, when it will change agent's status somehow.
     * @return List of all weapons. Note: Spawned items are included only.
     * @note Returns copies of the weapon infos to prevent outer modifications.
     */
    @SuppressWarnings ("unchecked")
    public ArrayList<Weapon> getAllWeapons (boolean pickableOnly, boolean greedy)
    {
        return (ArrayList) getItems (weapons.all, true, pickableOnly, greedy);
    }

    /**
     * Retreives list of all ammos (both known and thrown).
     * @param pickableOnly If true, the list will be filtered by pickability.
     * An item is considered pickable, when the agent can pick it up.
     * @param greedy If false, the list will be filtered by usefullness.
     * An item is considered useful, when it will change agent's status somehow.
     * @return List of all ammos. Note: Spawned items are included only.
     * @note Returns copies of the ammo infos to prevent outer modifications.
     */
    @SuppressWarnings ("unchecked")
    public ArrayList<Ammo> getAllAmmos (boolean pickableOnly, boolean greedy)
    {
        return (ArrayList) getItems (ammos.all, true, pickableOnly, greedy);
    }

    /**
     * Retreives list of all healths (both known and thrown).
     * @param pickableOnly If true, the list will be filtered by pickability.
     * An item is considered pickable, when the agent can pick it up.
     * @param greedy If false, the list will be filtered by usefullness.
     * An item is considered useful, when it will change agent's status somehow.
     * @return List of all healths. Note: Spawned items are included only.
     * @note Returns copies of the health infos to prevent outer modifications.
     */
    @SuppressWarnings ("unchecked")
    public ArrayList<Health> getAllHealths (boolean pickableOnly, boolean greedy)
    {
        return (ArrayList) getItems (healths.all, true, pickableOnly, greedy);
    }

    /**
     * Retreives list of all armors (both known and thrown).
     * @param pickableOnly If true, the list will be filtered by pickability.
     * An item is considered pickable, when the agent can pick it up.
     * @param greedy If false, the list will be filtered by usefullness.
     * An item is considered useful, when it will change agent's status somehow.
     * @return List of all armors. Note: Spawned items are included only.
     * @note Returns copies of the armor infos to prevent outer modifications.
     */
    @SuppressWarnings ("unchecked")
    public ArrayList<Armor> getAllArmors (boolean pickableOnly, boolean greedy)
    {
        return (ArrayList) getItems (armors.all, true, pickableOnly, greedy);
    }

    /**
     * Retreives list of all specials (both known and thrown).
     * @param pickableOnly If true, the list will be filtered by pickability.
     * An item is considered pickable, when the agent can pick it up.
     * @param greedy If false, the list will be filtered by usefullness.
     * An item is considered useful, when it will change agent's status somehow.
     * @return List of all specials. Note: Spawned items are included only.
     * @note Returns copies of the special infos to prevent outer modifications.
     */
    @SuppressWarnings ("unchecked")
    public ArrayList<Special> getAllSpecials (boolean pickableOnly, boolean greedy)
    {
        return (ArrayList) getItems (specials.all, true, pickableOnly, greedy);
    }

    /*========================================================================*/

    /**
     * Retreives list of all item pickup points.
     * @param pickableOnly If true, the list will be filtered by pickability.
     * An item is considered pickable, when the agent can pick it up.
     * @param greedy If false, the list will be filtered by usefullness.
     * An item is considered useful, when it will change agent's status somehow.
     * @return List of all items. Note: Empty pickups are included as well.
     * @note Returns copies of the item infos to prevent outer modifications.
     */
    public ArrayList<Item> getKnownPickups (boolean pickableOnly, boolean greedy)
    {
        return getItems (items.known, false, pickableOnly, greedy);
    }

    /*========================================================================*/

    /**
     * Retreives list of items from given container.
     * @param container Container to retreive the items from.
     * @param spawnedOnly If true, the list will be filtered by pickup emptyness.
     * @param pickableOnly If true, the list will be filtered by pickability.
     * An item is considered pickable, when the agent can pick it up.
     * @param greedy If false, the list will be filtered by usefullness.
     * An item is considered useful, when it will change agent's status somehow.
     * @return List of all items.
     * @note Returns copies of the item infos to prevent outer modifications.
     */
    private synchronized ArrayList<Item> getItems (ItemsContainer container, boolean spawnedOnly, boolean pickableOnly, boolean greedy)
    {
        ArrayList<Item> result = new ArrayList<Item> (container.size ());
        // make deep clone of the items, filter rightaway
        for (Item i : container.values ())
        {
            // filter empty pickups..
            if (spawnedOnly && !isPickupSpawned (i.ID))
                continue;

            // filter by pickability and usefulness
            if ( (pickableOnly || !greedy) && !isItemUseful (i, greedy) )
                continue;

            // include clone of the item
            result.add((Item)i.clone ());
        }
        // return created list
        return result;
    }

    /*========================================================================*/

    /**
     * Retreives info about given navigation point.
     * @param ID ID of the navigation point to be retreived.
     * @return Requested navigation point; or null if none.
     */
    public synchronized NavPoint getNavPoint (int ID)
    {
        return navPoints.known.get(ID);
    }

    /**
     * Retreives info about given navigation point.
     * @param UnrealID UnrealID of the navigation point to be retreived.
     * @return Requested navigation point; or null if none.
     */
    public synchronized NavPoint getNavPoint (String UnrealID)
    {
        return navPoints.known.get(UnrealID);
    }

    /*========================================================================*/

    /**
     * Tells, whether the given pickup point contains a spawned item.
     * @param itemID ID of item, for which its pickup point is to be examined.
     * @return True, if the item is spawned; false if the pickup is empty.
     */
    public synchronized boolean isPickupSpawned (int itemID)
    {
        // is this item reported as unsynchronized?
        if (pickups.unsyncItems.contains(itemID))
            // well, prevent false report of being empty
            return true;

        // retreive associated navpoint
        NavPoint navPoint = pickups.getByItem (itemID);
        // if the navpoint is not visible, we know nothing about the pickup
        if ( (navPoint == null) || !navPoint.visible )
            // so, assume the item's there..
            return true;

        // retreive queried item
        Item item = items.visible.get(itemID);
        // if the item is not visible, but its navpoint is..
        if (item == null)
            // the pickup is empty as the agent can see..
            return false;

        // if the pickup is visible and the item as well..
        // there is nothing to be queried upon..
        return true;
    }

    /*========================================================================*/

    /**
     * Refreshes item info and adds the item into visible items.
     * @param specificItems Specific-type items.
     * @param item Item message to be relied upon.
     */
    private synchronized void addItem (ItemsInfo specificItems, Item item)
    {
        // add to all items
        items.all.put (item);
        specificItems.all.put (item);

        // does it have a navpoint?
        if (item.navPoint != null)
        {
            // add to the knowledge (or at least refresh the knowledge)
            items.known.put (item);
            specificItems.known.put (item);

            // note: all known items are being sent before the first spawn,
            // therefore we must check the spawned flag
            if (spawned)
            {
                // add to the visibles as well
                items.visible.put (item);
                specificItems.visible.put (item);
            }
        }
        // no navpoint meand thrown item
        // note: known items can be sent without their navpoints even before
        // the first agent spawn (and who knows why?!?), thus they would be
        // mistaken for thrown items.. as a fix to that, we check spawned flag
        else if (spawned)
        {
            // add to the visibles
            items.visible.put(item);
            specificItems.visible.put(item);
        }

        // the adding of item synchronizes adding of navpoint
        pickups.unsyncItems.remove(item.ID);
    }

    /**
     * Removes items from visible items of specific type.
     * @param specificItems Specific-type items.
     * @param item Item to be removed.
     */
    private synchronized void removeItem (ItemsInfo specificItems, Item item)
    {
        // remove from the visibles
        specificItems.visible.remove(item.ID);
        // is it a thrown item?
        if (item.navPoint == null)
        {
            // remove from all items
            specificItems.all.remove(item.ID);
        }
    }

    /**
     * Removes item from visible items.
     * @param msg Delete message to be relied upon.
     */
    private synchronized void removeItem (DeleteFromBatch msg)
    {
        // remove from visibles and check if it were removed
        Item item = items.visible.remove(msg.ID);
        if (item != null)
        {
            // update rachability for known
            item.reachable = false;

            // is it a thrown item?
            if (item.navPoint == null)
            {
                // remove from all items
                items.all.remove(item.ID);
            }

            // remove also from the specific map
            switch (item.type)
            {
                case WEAPON:
                    removeItem (weapons, item);
                    break;
                case AMMO:
                    removeItem (ammos, item);
                    break;
                case HEALTH:
                    removeItem (healths, item);
                    break;
                case ARMOR:
                    removeItem (armors, item);
                    break;
                case SPECIAL:
                    removeItem (specials, item);
                    break;
            }

            // if the removal of item comes before removal of navpoint
            if (pickups.containsItem (item.ID))
            {
                // unsynchronize this item for a while
                pickups.unsyncItems.add(item.ID);
            }
        }
    }

    /*========================================================================*/

    /**
     * Refreshes navpoint info and adds the navpoint into visible navpoints.
     * @param navPoint NavPoint message to be relied upon.
     */
    private synchronized void addNavPoint (NavPoint navPoint)
    {
        // add to all navpoints
        navPoints.known.put (navPoint);
        // note: all navpoints are being sent before the first spawn, therefore
        // we must check the spawned flag
        if (spawned && navPoint.visible)
        {
            // add to the visibles as well
            navPoints.visible.put (navPoint);
            // does it have an item?
            if (navPoint.itemID > 0)
            {
                // add to the visible pickups as well
                pickups.put (navPoint);

                // if the adding of navpoint comes before adding of item
                if (!items.visible.containsKey(navPoint.itemID))
                {
                    // unsynchronize the item for a while
                    pickups.unsyncItems.add(navPoint.itemID);
                }
            }
        }
    }

    /**
     * Removes navpoint from visible navpoints.
     * @param msg Delete message to be relied upon.
     */
    private synchronized void removeNavPoint (DeleteFromBatch msg)
    {
        // remove from visibles and check if it were removed
        NavPoint navPoint = navPoints.visible.remove(msg.ID);
        if (navPoint != null)
        {
            // is it a pickup point?
            if (navPoint.itemID > 0)
            {
                // remove from visible pickups
                pickups.remove(navPoint);

                // the removal of navpoint synchronizes removal of item
                pickups.unsyncItems.remove(navPoint.itemID);

            }
        }
    }

    /*========================================================================*/

    /**
     * Closes batch. Clears all assumed unsynchronization.
     */
    private synchronized void closeBatch ()
    {
        pickups.unsyncItems.clear ();
    }

    /*========================================================================*/

    /**
     * Determines, whether a weapon can be useful for the agent.
     *
     * <h4>Pogamut troubles</h4>
     *
     * Is the "weapon stay" game setting enabled or not? Right now, the Pogamut
     * does not provide this kind of information. At least not to my knowledge.
     * Therefore, we shall assume that it is turned off and that we can pick-up
     * weapons we already carry in order to refill ammo.
     *
     * @param weapon Given weapon to be checked.
     * @param greedy Whether to pick-up items that will not do us any good.
     * @return True, if the weapon can be useful.
     */
    public boolean isWeaponUseful (Weapon weapon, boolean greedy)
    {
        // is the "weapon stay" game setting enabled?
        // if so, the ammo can not be refilled from weapons
        if (/* weapon stay is turned off */false)
        {
            // weapons we already have are not even pickable
            // unless, they are thrown by other players (well, mostly)
            return
                (weapon.navPoint == null)
                || !memory.inventory.hasWeapon (weapon.weaponType);
        }

        // get such weapon from inventory
        AddWeapon w = memory.inventory.getWeapon(weapon.weaponType);

        // do we have this weapon?
        if (w == null)
            return true;

        // is there some ammo space left?
        if (w.currentAmmo < w.maxAmmo)
            return true;

        // well, we have the a like this and enough ammo..
        return greedy;
    }

    /**
     * Determines, whether an ammo can be useful for the agent.
     *
     * @param ammo Given ammo to be checked.
     * @param greedy Whether to pick-up items that will not do us any good.
     * @return True, if the ammo can be useful.
     */
    public boolean isAmmoUseful (Ammo ammo, boolean greedy)
    {
        // get such weapon from inventory
        AddWeapon w = memory.inventory.getWeapon(ammo.typeOfWeapon);

        // do we have this weapon?
        if (w == null)
            return false;

        // is there some ammo space left?
        if (w.currentAmmo < w.maxAmmo)
            return true;

        // well, we're full of ammo like this..
        return false;
    }

    /**
     * Determines, whether a health can be useful for the agent.
     *
     * @param health Given health to be checked.
     * @param greedy Whether to pick-up items that will not do us any good.
     * @return True, if the health can be useful.
     */
    public boolean isHealthUseful (Health health, boolean greedy)
    {
        // if the health is boostable, pick it up..
        if (health.boostable)
        {
            // boostable health is always pickable
            if (greedy)
                return true;

            // will this health make any difference?
            return memory.self.getHealth () < 199;
        }

        // will this health make any difference?
        return memory.self.getHealth () < 100;
    }

    /**
     * Determines, whether an armor can be useful for the agent.
     *
     * <h4>Pogamut troubles</h4>
     *
     * AFAIK, there are two different types of shields: small one, which is
     * worth 50 armor points and big one with 100 points. Now, the agent can
     * have armor of upto 150 points. However, picking up two small shields
     * does not yield 100 armor points, but only 50. And the big shield works
     * the same way. Therefore, the agent needs to pickup both types to reach
     * 150 point.
     *
     * <p>Well. Now, how are we supposed to decide, whether we want some
     * specific shield, if we do not know, what kind of armor we already have?
     * Since Pogamut does not provide info on these two types separately, we
     * are going to do it by common guess.</p>
     *
     * @param armor Given armor to be checked.
     * @param greedy Whether to pick-up items that will not do us any good.
     * @return True, if the armor can be useful.
     */
    public boolean isArmorUseful (Armor armor, boolean greedy)
    {
        // armor is always pickable
        if (greedy)
            return true;

        // retreive current agent armor info
        int agentArmor = memory.self.getArmor ();

        // if the shield strength is the same as our armor
        if (agentArmor == armor.strenght)
            // then we assume we already have this kind of shield
            // and thus it would not help us much..
            return false;

        // now, if we're not full of all types of shields
        // we shall assume that this one can actually help us
        return agentArmor < 150;
    }

    /**
     * Determines, whether a special can be useful for the agent.
     *
     * @param special Given special to be checked.
     * @param greedy Whether to pick-up items that will not do us any good.
     * @return True, if the special can be useful.
     */
    public boolean isSpecialUseful (Special special, boolean greedy)
    {
        // special is always pickable
        if (greedy)
            return true;

        // is it a damage amplifier?
        if (special.typeOfSpecial == ItemType.U_DAMAGE_PACK)
            return true;

        return false;
    }

    /**
     * Determines, whether an item can be useful for the agent.
     *
     * @param item Given item to be checked.
     * @param greedy Whether to pick-up items that will not do us any good.
     * @return True, if the item can be useful.
     */
    public boolean isItemUseful (Item item, boolean greedy)
    {
        switch (item.type)
        {
            case WEAPON:
                return isWeaponUseful((Weapon)item, greedy);
            case AMMO:
                return isAmmoUseful((Ammo)item, greedy);
            case HEALTH:
                return isHealthUseful((Health)item, greedy);
            case ARMOR:
                return isArmorUseful((Armor)item, greedy);
            case SPECIAL:
                return isSpecialUseful((Special)item, greedy);
            default:
                return false;
        }
    }

    /*========================================================================*/

    /**
     * Tells, whether the agent spawned into the game already. Divides between
     * receiving full list of <i>known</i> items and casual <i>seen</i> items.
     */
    private boolean spawned = false;

    /**
     * Sets {@link #spawned} flag to true. Used upon spawn message.
     */
    private void setSpawn ()
    {
        spawned = true;
    }

    /*========================================================================*/

    /**
     * Info about items.
     */
    private class ItemsContainer
    {
        /** Items mapped by their IDs. */
        private HashMap<Integer, Item> byIDs = new HashMap<Integer, Item> ();
        /** Items mapped by their UnrealIDs. */
        private HashMap<String, Item> byUnrealIDs = new HashMap<String, Item> ();

        /*====================================================================*/

        /**
         * Tells, whether given item is present in the container.
         * @param ID ID of the item to be examined.
         * @return True, if queried item is present.
         */
        protected boolean containsKey (int ID)
        {
            return byIDs.containsKey(ID);
        }

        /**
         * Retreives given item from the container.
         * @param ID ID of the item to be retreived.
         * @return Requested item; or null, if none.
         */
        protected Item get (int ID)
        {
            return byIDs.get(ID);
        }

        /**
         * Retreives given item from the container.
         * @param UnrealID UnrealID of the item to be retreived.
         * @return Requested item; or null, if none.
         */
        protected Item get (String UnrealID)
        {
            return byUnrealIDs.get(UnrealID);
        }

        /**
         * Adds given item to the container.
         * @param item Item to be added.
         */
        protected void put (Item item)
        {
            byIDs.put(item.ID, item);
            byUnrealIDs.put(item.UnrealID, item);
        }

        /**
         * Removes given item from the container.
         * @param ID ID of the item to be removed.
         * @return Item being removed; or null, if none.
         */
        protected Item remove (int ID)
        {
            Item i = byIDs.remove (ID);
            if (null != i)
                return byUnrealIDs.remove (i.UnrealID);
            return null;
        }

        /**
         * Size of the container.
         * @return Size of the container.
         */
        protected int size ()
        {
            return byIDs.size ();
        }

        /**
         * Retreives all stored items from the container.
         * @return Collection of all stored items.
         */
        protected Collection<Item> values()
        {
            return byIDs.values ();
        }
    }

    /**
     * Info about specific type of items.
     */
    private class ItemsInfo
    {
        /** Known items. */
        protected ItemsContainer known = new ItemsContainer ();
        /** Visible items. */
        protected ItemsContainer visible = new ItemsContainer ();
        /** Union of known and visible items. */
        protected ItemsContainer all = new ItemsContainer ();
    }

    /*========================================================================*/

    /**
     * Info about specific type of navpoints.
     */
    private class NavPointsContainer
    {
        /** Navpoints mapped by their IDs. */
        private HashMap<Integer, NavPoint> byIDs = new HashMap<Integer, NavPoint> ();
        /** Navpoints mapped by their UnrealIDs. */
        private HashMap<String, NavPoint> byUnrealIDs = new HashMap<String, NavPoint> ();

        /*====================================================================*/

        /**
         * Tells, whether given navpoint is present in the container.
         * @param ID ID of the navpoint to be examined.
         * @return True, if queried navpoint is present.
         */
        protected boolean containsKey (int ID)
        {
            return byIDs.containsKey(ID);
        }

        /**
         * Retreives given navpoint from the container.
         * @param ID ID of the navpoint to be retreived.
         * @return Requested navpoint; or null, if none.
         */
        protected NavPoint get (int ID)
        {
            return byIDs.get(ID);
        }

        /**
         * Retreives given navpoint from the container.
         * @param UnrealID UnrealID of the navpoint to be retreived.
         * @return Requested navpoint; or null, if none.
         */
        protected NavPoint get (String UnrealID)
        {
            return byUnrealIDs.get(UnrealID);
        }

        /**
         * Adds given navpoint to the container.
         * @param navPoint NavPoint to be added.
         */
        protected void put (NavPoint navPoint)
        {
            byIDs.put(navPoint.ID, navPoint);
            byUnrealIDs.put(navPoint.UnrealID, navPoint);
        }

        /**
         * Removes given navpoint from the container.
         * @param ID ID of the navpoint.
         * @return NavPoint being removed; or null, if none.
         */
        protected NavPoint remove (int ID)
        {
            NavPoint np = byIDs.remove (ID);
            if (null != np)
                return byUnrealIDs.remove (np.UnrealID);
            return null;
        }

        /**
         * Size of the container.
         * @return Size of the container.
         */
        protected int size ()
        {
            return byIDs.size ();
        }

        /**
         * Retreives all stored navpoints from the container.
         * @return Collection of all stored navpoints.
         */
        protected Collection<NavPoint> values()
        {
            return byIDs.values ();
        }
    }

    /**
     * Info about specific type of navpoints.
     */
    private class NavPointsInfo
    {
        /** Known navpoints. */
        protected NavPointsContainer known = new NavPointsContainer ();
        /** Visible navpoints. */
        protected NavPointsContainer visible = new NavPointsContainer ();
    }

    /*========================================================================*/

    private class PickupsInfo
    {
        /** All visible pickups (inventory spots), mapped by IDs of their items. */
        private HashMap<Integer, NavPoint> byIDs = new HashMap<Integer, NavPoint> ();
        /** All visible pickups (inventory spots), mapped by UnrealIDs of their items. */
        private HashMap<String, NavPoint> byUnrealIDs = new HashMap<String, NavPoint> ();

        /**
         * Unsynchronized pickups. This set contains all pickups, where items
         * are already removed from visibles, while their navpoints are still
         * visible. The same applies when navpoints are visible, but items
         * are not.
         *
         * <p>Such mis-synchronization may arise during receiving of a message
         * batch, when item messages and navpoint messages are sent in wrong
         * sequence.</p>
         *
         * <p>This set is used when the agent asks, which pickups are heisted
         * (with no spawned item there). Whether a specific pickup currently
         * empty is decided on visibility of the associated navpoint and the
         * spawning item. To prevent false reporting of unsynchronized pickups
         * as empty, this set contains all pickups, which can not be answered
         * correctly.</p>
         *
         * <p>This set is automatically cleared at the end of each message
         * batch, since at that point, all pickups get synchronized no matter
         * what order they were sent in.</p>
         */
        protected HashSet<Integer> unsyncItems = new HashSet<Integer> ();

        /*====================================================================*/

        /**
         * Tells, whether pickup with given item is present in the container.
         * @param itemID ID of the item of the pickup to be examined.
         * @return True, if queried pickup is present. False otherwise.
         */
        protected boolean containsItem (int itemID)
        {
            return byIDs.containsKey(itemID);
        }

        /**
         * Retreives given pickup by its item from the container.
         * @param itemID ID of the item of the pickup to be removed.
         * @return Requested pickup navpoint; or null, if none.
         */
        protected NavPoint getByItem (int itemID)
        {
            return byIDs.get(itemID);
        }

        /**
         * Adds given pickup to the container.
         * @param navPoint Pickup to be added.
         */
        protected void put (NavPoint navPoint)
        {
            byIDs.put(navPoint.itemID, navPoint);
            byUnrealIDs.put(navPoint.itemUnrealID, navPoint);
        }

        /**
         * Removes given pickup from the container.
         * @param navPoint Pickup to be removed.
         */
        protected void remove (NavPoint navPoint)
        {
            if (null != byIDs.remove (navPoint.itemID))
                byUnrealIDs.remove (navPoint.itemID);
        }

        /**
         * Size of the container.
         * @return Size of the container.
         */
        protected int size ()
        {
            return byIDs.size ();
        }
    }

    /*========================================================================*/

    /**
     * Listening class for messages from engine.
     */
    private class Listener extends LoqueListener
    {
        /**
         * Agent just spawned into the game.
         * @param msg Message to handle.
         */
        private void msgSpawn (Spawn msg)
        {
            setSpawn ();
        }

        /**
         * Agent learned about weapon in the map.
         * @param msg Message to handle.
         */
        private void msgWeapon (Weapon msg)
        {
            // add to known and/or visible
            addItem (weapons, msg);
        }

        /**
         * Agent learned about ammo in the map.
         * @param msg Message to handle.
         */
        private void msgAmmo (Ammo msg)
        {
            // add to known and/or visible
            addItem (ammos, msg);
        }

        /**
         * Agent learned about health in the map.
         * @param msg Message to handle.
         */
        private void msgHealth (Health msg)
        {
            // add to known and/or visible
            addItem (healths, msg);
        }

        /**
         * Agent learned about armor in the map.
         * @param msg Message to handle.
         */
        private void msgArmor (Armor msg)
        {
            // add to known and/or visible
            addItem (armors, msg);
        }

        /**
         * Agent learned about special in the map.
         * @param msg Message to handle.
         */
        private void msgSpecial (Special msg)
        {
            // add to known and/or visible
            addItem (specials, msg);
        }

        /**
         * Agent learned about navigation point in the map.
         * @param msg Message to handle.
         */
        private void msgNavPoint (NavPoint msg)
        {
            addNavPoint (msg);
        }

        /**
         * Agent received message about end of synchronous batch.
         * @param msg Message to handle.
         */
        private void msgEnd (EndMessage msg)
        {
            closeBatch ();
        }

        /**
         * Info about loosing things from sight.
         * @param msg Message to handle.
         */
        private void msgDeleteFromBatch (DeleteFromBatch msg)
        {
            switch (msg.msgType)
            {
                case ITEM:
                    removeItem (msg);
                    return;
                case NAV_POINT:
                    removeNavPoint (msg);
                    return;
            }
        }

        /*========================================================================*/

        /**
         * Message switch.
         * @param msg Message to handle.
         */
        protected void processMessage (MessageObject msg)
        {
            switch (msg.type)
            {
                case SPAWN:
                    msgSpawn ((Spawn) msg);
                    return;
                case WEAPON:
                    msgWeapon ((Weapon) msg);
                    return;
                case AMMO:
                    msgAmmo ((Ammo) msg);
                    return;
                case HEALTH:
                    msgHealth ((Health) msg);
                    return;
                case ARMOR:
                    msgArmor ((Armor) msg);
                    return;
                case SPECIAL:
                    msgSpecial ((Special) msg);
                    return;
                case NAV_POINT:
                    msgNavPoint ((NavPoint) msg);
                    return;
                case END:
                    msgEnd ((EndMessage) msg);
                    return;
                case DELETE_FROM_BATCH:
                    msgDeleteFromBatch ((DeleteFromBatch) msg);
                    return;
            }
        }

        /**
         * Constructor: Signs up for listening.
         */
        private Listener ()
        {
            body.addTypedRcvMsgListener (this, MessageType.SPAWN);
            body.addTypedRcvMsgListener (this, MessageType.WEAPON);
            body.addTypedRcvMsgListener (this, MessageType.AMMO);
            body.addTypedRcvMsgListener (this, MessageType.HEALTH);
            body.addTypedRcvMsgListener (this, MessageType.ARMOR);
            body.addTypedRcvMsgListener (this, MessageType.SPECIAL);
            body.addTypedRcvMsgListener (this, MessageType.NAV_POINT);
            body.addTypedRcvMsgListener (this, MessageType.END);
            body.addTypedRcvMsgListener (this, MessageType.DELETE_FROM_BATCH);
        }
    }

    /** Listener. */
    private LoqueListener listener;

    /*========================================================================*/

    /** Agent's main. */
    protected Main main;
    /** Loque memory. */
    protected LoqueMemory memory;
    /** Agent's body. */
    protected AgentBody body;
    /** Agent's log. */
    protected Logger log;

    /*========================================================================*/

    /**
     * Constructor.
     * @param main Agent's main.
     * @param memory Loque memory.
     */
    public LoqueMapItems (Main main, LoqueMemory memory)
    {
        // setup reference to agent
        this.main = main;
        this.memory = memory;
        this.body = main.getBody ();
        this.log = main.getLogger ();

        // create listener
        this.listener = new Listener ();
    }
}